<?php

class QR_Jump_Public {
    private $plugin_name;
    private $version;
    private $qr_generator;
    private $analytics;

    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->qr_generator = new QR_Jump_QR_Generator();
        $this->analytics = new QR_Jump_Analytics();

        $this->init_hooks();
    }

    private function init_hooks() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_styles'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('init', array($this, 'register_shortcodes'));
        add_action('template_redirect', array($this, 'handle_qr_redirect'));
    }

    public function enqueue_styles() {
        wp_enqueue_style(
            $this->plugin_name,
            QR_JUMP_PLUGIN_URL . 'public/css/qr-jump-public.css',
            array(),
            $this->version,
            'all'
        );
    }

    public function enqueue_scripts() {
        wp_enqueue_script(
            $this->plugin_name,
            QR_JUMP_PLUGIN_URL . 'public/js/qr-jump-public.js',
            array('jquery'),
            $this->version,
            false
        );
    }

    public function register_shortcodes() {
        add_shortcode('qr_jump', array($this, 'qr_jump_shortcode'));
    }

    public function qr_jump_shortcode($atts) {
        $atts = shortcode_atts(array(
            'id' => 0,
            'title' => '',
            'show_count' => 'yes',
            'show_download' => 'yes',
            'size' => '',
            'color' => ''
        ), $atts, 'qr_jump');

        $post = get_post($atts['id']);
        if (!$post || $post->post_type !== 'qr_code') {
            return '';
        }

        $destination = get_post_meta($post->ID, '_qr_jump_destination', true);
        if (!$destination) {
            return '';
        }

        // Check if destination is vCard data
        $is_vcard = strpos(trim($destination), 'BEGIN:VCARD') === 0;

        $qr_url = home_url('qr/' . $post->post_name);
        $qr_options = array();

        if ($atts['size']) {
            $qr_options['size'] = intval($atts['size']);
        } else {
            $qr_options['size'] = get_post_meta($post->ID, '_qr_jump_size', true) ?: 300;
        }

        if ($atts['color']) {
            $qr_options['color'] = $atts['color'];
        } else {
            $qr_options['color'] = get_post_meta($post->ID, '_qr_jump_color', true) ?: '#000000';
        }

        $qr_image = $this->qr_generator->generate_qr_code($qr_url, $qr_options);
        if (!$qr_image) {
            return '';
        }

        $stats = $this->analytics->get_scan_stats($post->ID);
        $output = '<div class="qr-jump-container">';

        if ($atts['title']) {
            $output .= '<h3 class="qr-jump-title">' . esc_html($atts['title']) . '</h3>';
        }

        $output .= '<div class="qr-jump-code">';
        $output .= '<img src="' . esc_attr($qr_image) . '" alt="QR Code">';

        if ($atts['show_count'] === 'yes') {
            $output .= '<p class="qr-jump-scans">' . sprintf(
                _n('%s scan', '%s scans', $stats['total_scans'], 'qr-jump'),
                number_format_i18n($stats['total_scans'])
            ) . '</p>';
        }

        if ($atts['show_download'] === 'yes') {
            if ($is_vcard) {
                $output .= '<a href="' . esc_url($qr_url) . '" class="qr-jump-download">' . 
                          __('Download Contact', 'qr-jump') . '</a>';
            } else {
                $output .= '<a href="' . esc_url($qr_image) . '" download="qr-code.png" class="qr-jump-download">' . 
                          __('Download QR Code', 'qr-jump') . '</a>';
            }
        }

        $output .= '</div></div>';

        return $output;
    }

    public function handle_qr_redirect() {
        global $wp_query;

        if (!isset($wp_query->query_vars['qr_code'])) {
            return;
        }

        $slug = $wp_query->query_vars['qr_code'];
        $post = get_page_by_path($slug, OBJECT, 'qr_code');

        if (!$post || $post->post_status !== 'publish') {
            error_log('[QR Jump] Redirect handler: QR code not found for slug ' . $slug);
            wp_die(__('QR Code not found.', 'qr-jump'), '', array('response' => 404));
        }

        // Check expiry
        $expiry_date = get_post_meta($post->ID, '_qr_jump_expiry_date', true);
        if ($expiry_date && strtotime($expiry_date) < time()) {
            error_log('[QR Jump] Redirect handler: QR code expired for post ID ' . $post->ID);
            wp_die(__('This QR code has expired.', 'qr-jump'), '', array('response' => 410));
        }

        // Check scan limit
        $scan_limit = get_post_meta($post->ID, '_qr_jump_scan_limit', true);
        if ($scan_limit) {
            $stats = $this->analytics->get_scan_stats($post->ID);
            if ($stats['total_scans'] >= $scan_limit) {
                error_log('[QR Jump] Redirect handler: QR code scan limit reached for post ID ' . $post->ID);
                wp_die(__('This QR code has reached its scan limit.', 'qr-jump'), '', array('response' => 410));
            }
        }

        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
        $ip = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '';
        $referrer = isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : '';

        // Only log scans for visitors who are not logged in and not WordPress/curl/etc
        $ua = strtolower($user_agent);
        if (!is_user_logged_in() && strpos($ua, 'wordpress') === false && strpos($ua, 'curl') === false) {
            error_log('[QR Jump] Logging scan for post ID ' . $post->ID . ' | IP: ' . $ip . ' | UA: ' . $user_agent);
            $this->analytics->log_scan(
                $post->ID,
                $ip,
                $user_agent,
                $referrer
            );

            // --- Per-QR notification logic ---
            $notify_enabled = get_post_meta($post->ID, '_qr_jump_notify_enabled', true);
            if ($notify_enabled) {
                $notify_threshold = intval(get_post_meta($post->ID, '_qr_jump_notify_threshold', true)) ?: 10;
                $notify_emails = get_post_meta($post->ID, '_qr_jump_notify_emails', true);
                $notify_milestones = get_post_meta($post->ID, '_qr_jump_notify_milestones', true);
                $global_emails = get_option('qr_jump_notification_emails', get_option('admin_email'));
                $emails = $notify_emails ? $notify_emails : $global_emails;
                $to = array_map('trim', explode(',', $emails));
                $scan_stats = $this->analytics->get_scan_stats($post->ID);
                $total_scans = intval($scan_stats['total_scans']);
                // Threshold notification
                $last_notified = intval(get_post_meta($post->ID, '_qr_jump_last_notified_count', true));
                $scans_since = $total_scans - $last_notified;
                if ($scans_since >= $notify_threshold) {
                    $code_name = get_the_title($post->ID);
                    $subject = sprintf(__('QR Jump : Scan Notification for %s', 'qr-jump'), $code_name);
                    if ($notify_threshold == 1) {
                        $body = sprintf(__('You just got another scan for %s! This code now has %d scans.', 'qr-jump'), $code_name, $total_scans);
                    } else {
                        $body = sprintf(__('You just got another %d scans for %s! This code now has %d scans.', 'qr-jump'), $scans_since, $code_name, $total_scans);
                    }
                    $headers = array('From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>');
                    wp_mail($to, $subject, $body, $headers);
                    update_post_meta($post->ID, '_qr_jump_last_notified_count', $total_scans);
                }
                // Milestone notifications
                $milestones = array();
                if ($notify_milestones) {
                    foreach (explode(',', $notify_milestones) as $m) {
                        $m = intval(trim($m));
                        if ($m > 0) $milestones[] = $m;
                    }
                }
                $notified_milestones = get_post_meta($post->ID, '_qr_jump_notified_milestones', true);
                $notified_milestones = $notified_milestones ? explode(',', $notified_milestones) : array();
                foreach ($milestones as $milestone) {
                    if ($total_scans >= $milestone && !in_array($milestone, $notified_milestones)) {
                        $subject = __('QR Jump Milestone Notification', 'qr-jump');
                        $body = sprintf(__('QR code "%s" (%s) has reached the milestone of %d scans.', 'qr-jump'), get_the_title($post->ID), $post->ID, $milestone);
                        $headers = array('From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>');
                        wp_mail($to, $subject, $body, $headers);
                        $notified_milestones[] = $milestone;
                    }
                }
                update_post_meta($post->ID, '_qr_jump_notified_milestones', implode(',', $notified_milestones));
            }
            // --- End per-QR notification logic ---
        } else {
            error_log('[QR Jump] NOT logging scan (logged-in user or bot) for post ID ' . $post->ID . ' | IP: ' . $ip . ' | UA: ' . $user_agent);
        }

        // Get destination URL
        $destination = get_post_meta($post->ID, '_qr_jump_destination', true);
        if (!$destination) {
            error_log('[QR Jump] Redirect handler: No destination for post ID ' . $post->ID);
            wp_die(__('Invalid QR code destination.', 'qr-jump'), '', array('response' => 500));
        }

        // Check if destination is vCard data
        if (strpos(trim($destination), 'BEGIN:VCARD') === 0) {
            // Serve vCard data as downloadable file
            $filename = sanitize_file_name(get_the_title($post->ID) . '.vcf');
            
            // Set headers for vCard download
            header('Content-Type: text/vcard');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Content-Length: ' . strlen($destination));
            header('Cache-Control: no-cache, must-revalidate');
            header('Pragma: no-cache');
            
            // Output vCard data
            echo $destination;
            exit;
        }

        // Redirect to destination (URL)
        error_log('[QR Jump] Redirecting to ' . $destination . ' for post ID ' . $post->ID);
        wp_redirect($destination, 302);
        exit;
    }
} 