<?php

class QR_Jump_Updater {
    private static $instance = null;
    private $current_version;
    private $plugin_slug;
    private $plugin_file;

    public function __construct($plugin_file) {
        $this->plugin_file = $plugin_file;
        $this->plugin_slug = basename($plugin_file, '.php');
        $this->current_version = get_option('qr_jump_version', '1.0.0');
    }

    public static function get_instance($plugin_file) {
        if (null === self::$instance) {
            self::$instance = new self($plugin_file);
        }
        return self::$instance;
    }

    public function check_update() {
        // Only run update checks if we're in the admin
        if (!is_admin()) {
            return;
        }

        // Get the new version from the plugin file
        $plugin_data = get_plugin_data($this->plugin_file);
        $new_version = $plugin_data['Version'];

        // If versions match, no update needed
        if (version_compare($this->current_version, $new_version, '==')) {
            return;
        }

        // Run the update
        $this->update($new_version);
    }

    private function update($new_version) {
        global $wpdb;

        // Start transaction
        $wpdb->query('START TRANSACTION');

        try {
            // Update database tables if needed
            $this->update_database_tables();

            // Update version number
            update_option('qr_jump_version', $new_version);

            // Commit transaction
            $wpdb->query('COMMIT');

            // Clear any caches
            wp_cache_flush();
            
            // Flush rewrite rules
            flush_rewrite_rules();

        } catch (Exception $e) {
            // Rollback on error
            $wpdb->query('ROLLBACK');
            error_log('QR Jump update failed: ' . $e->getMessage());
        }
    }

    private function update_database_tables() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        // Table for QR code analytics
        $table_name = $wpdb->prefix . 'qr_jump_analytics';
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            qr_id bigint(20) NOT NULL,
            scan_date datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            ip_address varchar(45) NOT NULL,
            user_agent text NOT NULL,
            referrer text,
            country varchar(2),
            city varchar(100),
            latitude decimal(10,8),
            longitude decimal(11,8),
            PRIMARY KEY  (id),
            KEY qr_id (qr_id),
            KEY scan_date (scan_date)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
} 