<?php

use Endroid\QrCode\QrCode;
use Endroid\QrCode\Writer\PngWriter;
use Endroid\QrCode\Color\Color;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel\ErrorCorrectionLevelHigh;
use Endroid\QrCode\RoundBlockSizeMode\RoundBlockSizeModeMargin;
use Endroid\QrCode\Label\Label;
use Endroid\QrCode\Writer\SvgWriter;

class QR_Jump_QR_Generator {
    private $default_size = 300;
    private $default_color = '#000000';
    private $default_background = '#FFFFFF';

    public function __construct() {
        // Check if Composer autoload exists
        if (!file_exists(QR_JUMP_PLUGIN_DIR . 'vendor/autoload.php')) {
            add_action('admin_notices', array($this, 'missing_dependencies_notice'));
            return;
        }
        require_once QR_JUMP_PLUGIN_DIR . 'vendor/autoload.php';
    }

    public function missing_dependencies_notice() {
        ?>
        <div class="notice notice-error">
            <p><?php _e('QR Jump requires the endroid/qr-code library. Please run composer install in the plugin directory.', 'qr-jump'); ?></p>
        </div>
        <?php
    }

    private function hex_to_rgb($hex) {
        $hex = ltrim($hex, '#');
        if (strlen($hex) === 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }
        return array(
            'r' => hexdec(substr($hex, 0, 2)),
            'g' => hexdec(substr($hex, 2, 2)),
            'b' => hexdec(substr($hex, 4, 2))
        );
    }

    public function generate_qr_code($data, $options = array()) {
        if (!class_exists('Endroid\QrCode\QrCode')) {
            return false;
        }

        $size = isset($options['size']) ? $options['size'] : $this->default_size;
        $foreground = isset($options['color']) ? $options['color'] : $this->default_color;
        $background = isset($options['background']) ? $options['background'] : $this->default_background;

        // Convert hex colors to RGB
        $fg_rgb = $this->hex_to_rgb($foreground);
        $bg_rgb = $this->hex_to_rgb($background);

        $qrCode = new QrCode($data);
        $qrCode->setSize($size);
        $qrCode->setMargin(10);
        $qrCode->setForegroundColor(new Color($fg_rgb['r'], $fg_rgb['g'], $fg_rgb['b']));
        $qrCode->setBackgroundColor(new Color($bg_rgb['r'], $bg_rgb['g'], $bg_rgb['b']));
        $qrCode->setEncoding(new Encoding('UTF-8'));
        $qrCode->setErrorCorrectionLevel(new ErrorCorrectionLevelHigh());
        $qrCode->setRoundBlockSizeMode(new RoundBlockSizeModeMargin());

        if (isset($options['label'])) {
            $label = new Label($options['label']);
            $qrCode->setLabel($label);
        }

        $writer = new PngWriter();
        $result = $writer->write($qrCode);

        return $result->getDataUri();
    }

    public function generate_qr_code_file($data, $options = array(), $filename = null) {
        if (!class_exists('Endroid\QrCode\QrCode')) {
            return false;
        }

        $size = isset($options['size']) ? $options['size'] : $this->default_size;
        $foreground = isset($options['color']) ? $options['color'] : $this->default_color;
        $background = isset($options['background']) ? $options['background'] : $this->default_background;

        // Convert hex colors to RGB
        $fg_rgb = $this->hex_to_rgb($foreground);
        $bg_rgb = $this->hex_to_rgb($background);

        $qrCode = new QrCode($data);
        $qrCode->setSize($size);
        $qrCode->setMargin(10);
        $qrCode->setForegroundColor(new Color($fg_rgb['r'], $fg_rgb['g'], $fg_rgb['b']));
        $qrCode->setBackgroundColor(new Color($bg_rgb['r'], $bg_rgb['g'], $bg_rgb['b']));
        $qrCode->setEncoding(new Encoding('UTF-8'));
        $qrCode->setErrorCorrectionLevel(new ErrorCorrectionLevelHigh());
        $qrCode->setRoundBlockSizeMode(new RoundBlockSizeModeMargin());

        if (isset($options['label'])) {
            $label = new Label($options['label']);
            $qrCode->setLabel($label);
        }

        $writer = new PngWriter();
        $result = $writer->write($qrCode);

        if ($filename) {
            $result->saveToFile($filename);
            return true;
        }

        return $result->getString();
    }

    public function generate_qr_code_svg($url, $options = array()) {
        try {
            $size = isset($options['size']) ? $options['size'] : $this->default_size;
            $foreground = isset($options['color']) ? $options['color'] : $this->default_color;
            $background = isset($options['background']) ? $options['background'] : $this->default_background;

            $fg = $this->hex_to_rgb($foreground ?: '#000000');
            $bg = $this->hex_to_rgb($background ?: '#FFFFFF');

            $qrCode = new QrCode($url);
            $qrCode->setSize($size);
            $qrCode->setMargin(10);
            $qrCode->setForegroundColor(new Color($fg['r'], $fg['g'], $fg['b']));
            $qrCode->setBackgroundColor(new Color($bg['r'], $bg['g'], $bg['b']));
            $qrCode->setErrorCorrectionLevel(new ErrorCorrectionLevelHigh());
            $qrCode->setRoundBlockSizeMode(new RoundBlockSizeModeMargin());
            $qrCode->setEncoding(new Encoding('UTF-8'));

            $writer = new SvgWriter();
            $result = $writer->write($qrCode);
            return $result->getString();
        } catch (Exception $e) {
            error_log('QR Jump - SVG Generation Error: ' . $e->getMessage());
            return false;
        }
    }
} 