<?php

class QR_Jump_Analytics {
    private $table_name;

    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'qr_jump_analytics';
    }

    public function log_scan($qr_id, $ip_address, $user_agent, $referrer = '') {
        global $wpdb;

        $geo_data = $this->get_geo_data($ip_address);

        $data = array(
            'qr_id'      => $qr_id,
            'ip_address' => $ip_address,
            'user_agent' => $user_agent,
            'referrer'   => $referrer,
            'country'    => $geo_data['country'] ?? null,
            'city'       => $geo_data['city'] ?? null,
            'latitude'   => $geo_data['latitude'] ?? null,
            'longitude'  => $geo_data['longitude'] ?? null
        );

        return $wpdb->insert($this->table_name, $data);
    }

    public function get_scan_stats($qr_id) {
        global $wpdb;

        $stats = array(
            'total_scans' => 0,
            'unique_ips'  => 0,
            'countries'   => array(),
            'daily'       => array(),
            'monthly'     => array()
        );

        // Total scans
        $stats['total_scans'] = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->table_name} WHERE qr_id = %d",
            $qr_id
        ));

        // Unique IPs
        $stats['unique_ips'] = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT ip_address) FROM {$this->table_name} WHERE qr_id = %d",
            $qr_id
        ));

        // Countries
        $countries = $wpdb->get_results($wpdb->prepare(
            "SELECT country, COUNT(*) as count 
            FROM {$this->table_name} 
            WHERE qr_id = %d AND country IS NOT NULL 
            GROUP BY country",
            $qr_id
        ));

        foreach ($countries as $country) {
            $stats['countries'][$country->country] = $country->count;
        }

        // Daily scans
        $daily = $wpdb->get_results($wpdb->prepare(
            "SELECT DATE(scan_date) as date, COUNT(*) as count 
            FROM {$this->table_name} 
            WHERE qr_id = %d 
            GROUP BY DATE(scan_date) 
            ORDER BY date DESC 
            LIMIT 30",
            $qr_id
        ));

        foreach ($daily as $day) {
            $stats['daily'][$day->date] = $day->count;
        }

        // Monthly scans
        $monthly = $wpdb->get_results($wpdb->prepare(
            "SELECT DATE_FORMAT(scan_date, '%Y-%m') as month, COUNT(*) as count 
            FROM {$this->table_name} 
            WHERE qr_id = %d 
            GROUP BY month 
            ORDER BY month DESC 
            LIMIT 12",
            $qr_id
        ));

        foreach ($monthly as $month) {
            $stats['monthly'][$month->month] = $month->count;
        }

        return $stats;
    }

    public function export_csv($qr_id) {
        global $wpdb;

        $scans = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE qr_id = %d ORDER BY scan_date DESC",
            $qr_id
        ));

        if (empty($scans)) {
            return false;
        }

        $filename = 'qr-jump-scans-' . $qr_id . '-' . date('Y-m-d') . '.csv';
        $filepath = wp_upload_dir()['path'] . '/' . $filename;

        $fp = fopen($filepath, 'w');

        // Write headers
        fputcsv($fp, array(
            'Scan Date',
            'IP Address',
            'User Agent',
            'Referrer',
            'Country',
            'City',
            'Latitude',
            'Longitude'
        ));

        // Write data
        foreach ($scans as $scan) {
            fputcsv($fp, array(
                $scan->scan_date,
                $scan->ip_address,
                $scan->user_agent,
                $scan->referrer,
                $scan->country,
                $scan->city,
                $scan->latitude,
                $scan->longitude
            ));
        }

        fclose($fp);

        return $filepath;
    }

    // Get scans in the last day
    public function get_scans_last_day($qr_id) {
        global $wpdb;
        $since = date('Y-m-d H:i:s', strtotime('-1 day'));
        return (int) $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$this->table_name} WHERE qr_id = %d AND scan_date > %s", $qr_id, $since));
    }
    // Get scans in the last 7 days
    public function get_scans_last_7_days($qr_id) {
        global $wpdb;
        $since = date('Y-m-d H:i:s', strtotime('-7 days'));
        return (int) $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$this->table_name} WHERE qr_id = %d AND scan_date > %s", $qr_id, $since));
    }
    // Get scans in the last 30 days
    public function get_scans_last_30_days($qr_id) {
        global $wpdb;
        $since = date('Y-m-d H:i:s', strtotime('-30 days'));
        return (int) $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$this->table_name} WHERE qr_id = %d AND scan_date > %s", $qr_id, $since));
    }
    // Get scans this month
    public function get_scans_this_month($qr_id) {
        global $wpdb;
        $first_of_month = date('Y-m-01 00:00:00');
        return (int) $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$this->table_name} WHERE qr_id = %d AND scan_date >= %s", $qr_id, $first_of_month));
    }

    private function get_geo_data($ip_address) {
        // Skip local IPs
        if (in_array($ip_address, array('127.0.0.1', '::1'))) {
            return array();
        }

        $api_key = get_option('qr_jump_ipapi_key', '');
        if (empty($api_key)) {
            return array();
        }

        $response = wp_remote_get("https://ipapi.co/{$ip_address}/json/?key={$api_key}");

        if (is_wp_error($response)) {
            return array();
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (empty($data) || isset($data['error'])) {
            return array();
        }

        return array(
            'country'   => $data['country_code'] ?? null,
            'city'      => $data['city'] ?? null,
            'latitude'  => $data['latitude'] ?? null,
            'longitude' => $data['longitude'] ?? null
        );
    }
} 