<?php

class QR_Jump_Admin {
    private $plugin_name;
    private $version;
    private $qr_generator;
    private $analytics;

    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->qr_generator = new QR_Jump_QR_Generator();
        $this->analytics = new QR_Jump_Analytics();

        $this->init_hooks();
        add_action('all_admin_notices', array(
            $this,
            'show_logo_on_qr_code_list'
        ));
        add_action('all_admin_notices', array(
            $this,
            'show_logo_on_qr_code_edit'
        ));
    }

    private function init_hooks() {
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));
        add_action('save_post_qr_code', array($this, 'save_meta_box_data'));
        add_filter('manage_qr_code_posts_columns', array($this, 'set_custom_columns'));
        add_action('manage_qr_code_posts_custom_column', array($this, 'custom_column_content'), 10, 2);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_styles'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('add_meta_boxes', array($this, 'add_analytics_meta_box'));
        add_action('wp_ajax_qr_jump_download_qr', array($this, 'handle_qr_download'));
        add_action('admin_menu', array($this, 'add_settings_page'));
        add_action('admin_init', array($this, 'register_settings'));
        // Schedule cron event
        add_action('init', array($this, 'schedule_cron'));
        add_action('qr_jump_cron_summary', array($this, 'send_cron_summary'));
        add_action('wp_ajax_qr_jump_send_test_email', array($this, 'ajax_send_test_email'));
    }

    public function schedule_cron() {
        if (!wp_next_scheduled('qr_jump_cron_summary')) {
            wp_schedule_event(time(), 'daily', 'qr_jump_cron_summary');
        }
    }

    public function send_cron_summary() {
        // Check if notifications are enabled
        $enabled = get_option('qr_jump_notifications_enabled', '0');
        if ($enabled !== '1') return;
        $emails = get_option('qr_jump_notification_emails', get_option('admin_email'));
        $to = array_map('trim', explode(',', $emails));
        $frequency = get_option('qr_jump_notification_frequency', 'weekly');
        $email_format = get_option('qr_jump_notification_email_format', 'html');
        $last_sent = get_option('qr_jump_last_summary_sent', 0);
        $now = time();
        // Determine if it's the right day
        $send = false;
        if ($frequency === 'daily') {
            $send = true;
        } elseif ($frequency === 'weekly') {
            $send = strtolower(date('l', $now)) === 'monday';
        } elseif ($frequency === 'monthly') {
            $send = date('j', $now) == 1;
        }
        if (!$send) return;
        // Get all QR codes
        $args = array('post_type' => 'qr_code', 'post_status' => 'publish', 'numberposts' => -1);
        $qrs = get_posts($args);
        $rows = array();
        foreach ($qrs as $qr) {
            $analytics = new QR_Jump_Analytics();
            $stats = $analytics->get_scan_stats($qr->ID);
            $scan_count = 0;
            // Count scans since last report
            global $wpdb;
            $table = $wpdb->prefix . 'qr_jump_analytics';
            $since = $last_sent ? date('Y-m-d H:i:s', $last_sent) : '1970-01-01 00:00:00';
            $scan_count = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table WHERE qr_id = %d AND scan_date > %s", $qr->ID, $since));
            $rows[] = array(
                'title' => get_the_title($qr->ID),
                'id' => $qr->ID,
                'scans' => $scan_count
            );
        }
        // Build email
        $subject = __('QR Jump Scan Summary', 'qr-jump');
        if ($email_format === 'html') {
            $body = '<h2>' . esc_html__('QR Jump Scan Summary', 'qr-jump') . '</h2>';
            $body .= '<table border="1" cellpadding="5" cellspacing="0"><tr><th>' . esc_html__('QR Code', 'qr-jump') . '</th><th>' . esc_html__('ID', 'qr-jump') . '</th><th>' . esc_html__('Scans Since Last Report', 'qr-jump') . '</th></tr>';
            foreach ($rows as $row) {
                $body .= '<tr><td>' . esc_html($row['title']) . '</td><td>' . esc_html($row['id']) . '</td><td>' . esc_html($row['scans']) . '</td></tr>';
            }
            $body .= '</table>';
        } else {
            $body = "QR Jump Scan Summary\n\n";
            foreach ($rows as $row) {
                $body .= sprintf("%s (ID: %s): %d scans\n", $row['title'], $row['id'], $row['scans']);
            }
        }
        $headers = array('From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>');
        if ($email_format === 'html') {
            $headers[] = 'Content-Type: text/html; charset=UTF-8';
        }
        wp_mail($to, $subject, $body, $headers);
        update_option('qr_jump_last_summary_sent', $now);
    }

    public function add_settings_page() {
        add_submenu_page(
            'edit.php?post_type=qr_code',
            __('QR Jump Settings', 'qr-jump'),
            __('Settings', 'qr-jump'),
            'manage_options',
            'qr-jump-settings',
            array($this, 'render_settings_page')
        );
    }

    public function render_settings_page() {
        // Handle test email
        if (isset($_POST['qr_jump_send_test_email']) && check_admin_referer('qr_jump_test_email')) {
            $emails = isset($_POST['qr_jump_notification_emails']) && trim($_POST['qr_jump_notification_emails']) !== '' ? $_POST['qr_jump_notification_emails'] : get_option('admin_email');
            $format = isset($_POST['qr_jump_notification_email_format']) ? $_POST['qr_jump_notification_email_format'] : 'html';
            $to = array_map('trim', explode(',', $emails));
            $subject = __('QR Jump Test Email', 'qr-jump');
            $body = __('This is a test email from the QR Jump plugin. If you received this, notifications are working!', 'qr-jump');
            $headers = array('From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>');
            if ($format === 'html') {
                $headers[] = 'Content-Type: text/html; charset=UTF-8';
            }
            $sent = wp_mail($to, $subject, $body, $headers);
            if ($sent) {
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Test email sent successfully.', 'qr-jump') . '</p></div>';
            } else {
                echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Failed to send test email.', 'qr-jump') . '</p></div>';
            }
        }
        // Get current values or defaults
        $enabled = get_option('qr_jump_notifications_enabled', '0');
        $emails = get_option('qr_jump_notification_emails', get_option('admin_email'));
        $frequency = get_option('qr_jump_notification_frequency', 'weekly');
        $email_format = get_option('qr_jump_notification_email_format', 'html');
        echo '<div class="wrap"><h1>' . esc_html__('QR Jump Settings', 'qr-jump') . '</h1>';
        echo '<form method="post" action="options.php">';
        settings_fields('qr_jump_settings_group');
        do_settings_sections('qr-jump-settings');
        echo '<table class="form-table">';
        echo '<tr valign="top"><th scope="row">' . esc_html__('Enable Email Notifications', 'qr-jump') . '</th>';
        echo '<td><input type="checkbox" name="qr_jump_notifications_enabled" value="1"' . checked('1', $enabled, false) . '> ' . esc_html__('Enable', 'qr-jump') . '</td></tr>';
        echo '<tr valign="top"><th scope="row">' . esc_html__('Notification Email(s)', 'qr-jump') . '</th>';
        echo '<td><input type="text" name="qr_jump_notification_emails" value="' . esc_attr($emails) . '" class="regular-text"> <br><span class="description">' . esc_html__('Comma-separated. Defaults to admin email.', 'qr-jump') . '</span></td></tr>';
        echo '<tr valign="top"><th scope="row">' . esc_html__('Frequency', 'qr-jump') . '</th>';
        echo '<td>';
        echo '<label><input type="radio" name="qr_jump_notification_frequency" value="daily"' . checked('daily', $frequency, false) . '> ' . esc_html__('Daily', 'qr-jump') . '</label> ';
        echo '<label><input type="radio" name="qr_jump_notification_frequency" value="weekly"' . checked('weekly', $frequency, false) . '> ' . esc_html__('Weekly (Monday)', 'qr-jump') . '</label> ';
        echo '<label><input type="radio" name="qr_jump_notification_frequency" value="monthly"' . checked('monthly', $frequency, false) . '> ' . esc_html__('Monthly (1st)', 'qr-jump') . '</label>';
        echo '</td></tr>';
        echo '<tr valign="top"><th scope="row">' . esc_html__('Email Format', 'qr-jump') . '</th>';
        echo '<td>';
        echo '<label><input type="radio" name="qr_jump_notification_email_format" value="html"' . checked('html', $email_format, false) . '> ' . esc_html__('HTML', 'qr-jump') . '</label> ';
        echo '<label><input type="radio" name="qr_jump_notification_email_format" value="plain"' . checked('plain', $email_format, false) . '> ' . esc_html__('Plain Text', 'qr-jump') . '</label>';
        echo '</td></tr>';
        echo '</table>';
        submit_button();
        echo '</form>';
        // Place Send Test Email form immediately after the main form
        echo '<form method="post" action="" style="margin-top:10px;">';
        wp_nonce_field('qr_jump_test_email');
        echo '<input type="hidden" name="qr_jump_notification_emails" value="' . esc_attr($emails) . '">';
        echo '<input type="hidden" name="qr_jump_notification_email_format" value="' . esc_attr($email_format) . '">';
        echo '<button type="submit" name="qr_jump_send_test_email" value="1" class="button">' . esc_html__('Send Test Email', 'qr-jump') . '</button>';
        echo '</form>';
        // Inline CSS to hide day rows by default
        echo '<style>#qr-jump-weekly-row, #qr-jump-monthly-row { display: none; }</style>';
        echo '</div>';
    }

    // Register settings
    public function register_settings() {
        register_setting('qr_jump_settings_group', 'qr_jump_notifications_enabled');
        register_setting('qr_jump_settings_group', 'qr_jump_notification_emails');
        register_setting('qr_jump_settings_group', 'qr_jump_notification_frequency');
        register_setting('qr_jump_settings_group', 'qr_jump_notification_email_format');
    }

    public function enqueue_styles() {
        wp_enqueue_style(
            $this->plugin_name,
            QR_JUMP_PLUGIN_URL . 'admin/css/qr-jump-admin.css',
            array(),
            $this->version,
            'all'
        );
    }

    public function enqueue_scripts() {
        wp_enqueue_script('jquery');
        wp_enqueue_script(
            $this->plugin_name,
            QR_JUMP_PLUGIN_URL . 'admin/js/qr-jump-admin.js',
            array('jquery'),
            $this->version,
            false
        );

        wp_localize_script($this->plugin_name, 'qrJumpAdmin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('qr_jump_admin_nonce')
        ));
    }

    // AJAX handler for test email
    public function ajax_send_test_email() {
        check_ajax_referer('qr_jump_admin_nonce', 'nonce');
        $emails = isset($_POST['emails']) && trim($_POST['emails']) !== '' ? sanitize_text_field($_POST['emails']) : get_option('admin_email');
        $format = isset($_POST['format']) ? sanitize_text_field($_POST['format']) : 'html';
        $to = array_map('trim', explode(',', $emails));
        $subject = __('QR Jump Test Email', 'qr-jump');
        $body = __('This is a test email from the QR Jump plugin. If you received this, notifications are working!', 'qr-jump');
        $headers = array('From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>');
        if ($format === 'html') {
            $headers[] = 'Content-Type: text/html; charset=UTF-8';
        }
        $sent = wp_mail($to, $subject, $body, $headers);
        if ($sent) {
            wp_send_json_success(__('Test email sent successfully.', 'qr-jump'));
        } else {
            wp_send_json_error(__('Failed to send test email.', 'qr-jump'));
        }
    }

    public function add_meta_boxes() {
        add_meta_box(
            'qr_jump_details',
            __('QR Code Details', 'qr-jump'),
            array($this, 'render_meta_box'),
            'qr_code',
            'normal',
            'high'
        );
        add_meta_box(
            'qr_jump_analytics',
            __('Scan Analytics', 'qr-jump'),
            array($this, 'render_analytics_meta_box'),
            'qr_code',
            'normal',
            'high'
        );
    }

    public function render_meta_box($post) {
        wp_nonce_field('qr_jump_meta_box', 'qr_jump_meta_box_nonce');

        $destination = get_post_meta($post->ID, '_qr_jump_destination', true);
        $expiry_date = get_post_meta($post->ID, '_qr_jump_expiry_date', true);
        $scan_limit = get_post_meta($post->ID, '_qr_jump_scan_limit', true);
        $qr_color = get_post_meta($post->ID, '_qr_jump_color', true) ?: '#000000';
        $qr_size = get_post_meta($post->ID, '_qr_jump_size', true) ?: '300';
        $qr_url = home_url('qr/' . $post->post_name);

        // Convert expiry date to datetime-local format if set
        $expiry_date_local = '';
        if (!empty($expiry_date)) {
            // Assume expiry_date is stored in UTC, convert to local time using WordPress timezone
            $utc = new DateTime($expiry_date, new DateTimeZone('UTC'));
            $wp_tz = function_exists('wp_timezone') ? wp_timezone() : new DateTimeZone(wp_timezone_string());
            $utc->setTimezone($wp_tz);
            $expiry_date_local = $utc->format('Y-m-d\TH:i');
        }

        // Per-QR notification settings
        $notify_enabled = get_post_meta($post->ID, '_qr_jump_notify_enabled', true);
        $notify_threshold = get_post_meta($post->ID, '_qr_jump_notify_threshold', true) ?: 10;
        $notify_emails = get_post_meta($post->ID, '_qr_jump_notify_emails', true);
        $global_emails = get_option('qr_jump_notification_emails', get_option('admin_email'));

        ?>
        <div class="qr-jump-meta-box">
            <!-- Logo removed from here -->
            <p>
                <label for="qr_jump_destination"><?php _e('Destination (URL or vCard)', 'qr-jump'); ?></label>
                <textarea id="qr_jump_destination" name="qr_jump_destination" class="widefat" rows="4" required><?php echo esc_textarea($destination); ?></textarea>
                <span class="description"><?php _e('Enter a URL to redirect to, or vCard data for contact information', 'qr-jump'); ?></span>
            </p>

            <p>
                <label for="qr_jump_expiry_date"><?php _e('Expiry Date', 'qr-jump'); ?></label>
                <input type="datetime-local" id="qr_jump_expiry_date" name="qr_jump_expiry_date" value="<?php echo esc_attr($expiry_date_local); ?>">
            </p>

            <p>
                <label for="qr_jump_scan_limit"><?php _e('Scan Limit', 'qr-jump'); ?></label>
                <input type="number" id="qr_jump_scan_limit" name="qr_jump_scan_limit" value="<?php echo esc_attr($scan_limit); ?>" min="0">
                <span class="description"><?php _e('Leave empty for unlimited scans', 'qr-jump'); ?></span>
            </p>

            <p>
                <label for="qr_jump_color"><?php _e('QR Code Color', 'qr-jump'); ?></label>
                <input type="color" id="qr_jump_color" name="qr_jump_color" value="<?php echo esc_attr($qr_color); ?>">
            </p>

            <p>
                <label for="qr_jump_size"><?php _e('QR Code Size', 'qr-jump'); ?></label>
                <input type="number" id="qr_jump_size" name="qr_jump_size" value="<?php echo esc_attr($qr_size); ?>" min="100" max="1000">
                <span class="description"><?php _e('Size in pixels (100-1000)', 'qr-jump'); ?></span>
            </p>

            <p>
                <label><?php _e('QR Code URL', 'qr-jump'); ?></label>
                <input type="text" value="<?php echo esc_attr($qr_url); ?>" readonly style="width:100%;font-family:monospace;">
            </p>

            <?php if ($post->post_status === 'publish' && $destination) : ?>
                <div class="qr-code-preview">
                    <h3><?php _e('QR Code Preview', 'qr-jump'); ?></h3>
                    <?php
                    $qr_code_png = $this->qr_generator->generate_qr_code($qr_url, array('color' => $qr_color, 'size' => $qr_size));
                    if ($qr_code_png) {
                        echo '<img src="' . esc_attr($qr_code_png) . '" alt="QR Code Preview">';
                    }
                    ?>
                    <div class="qr-code-actions">
                        <div class="download-options">
                            <span class="download-label"><?php _e('Download as:', 'qr-jump'); ?></span>
                            <a href="<?php echo esc_url(admin_url('admin-ajax.php?action=qr_jump_download_qr&qr_id=' . $post->ID . '&format=png&nonce=' . wp_create_nonce('qr_jump_download'))); ?>" class="button"><?php _e('PNG', 'qr-jump'); ?></a>
                            <a href="<?php echo esc_url(admin_url('admin-ajax.php?action=qr_jump_download_qr&qr_id=' . $post->ID . '&format=svg&nonce=' . wp_create_nonce('qr_jump_download'))); ?>" class="button"><?php _e('SVG', 'qr-jump'); ?></a>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <hr><h4><?php _e('Notification Settings for this QR Code', 'qr-jump'); ?></h4>
            <p><label><input type="checkbox" name="qr_jump_notify_enabled" value="1"<?php checked('1', $notify_enabled, true); ?>> <?php _e('Enable scan notifications for this QR', 'qr-jump'); ?></label></p>
            <p><label><?php _e('Notify every', 'qr-jump'); ?> <input type="number" name="qr_jump_notify_threshold" value="<?php echo esc_attr($notify_threshold); ?>" min="1" style="width:60px;"> <?php _e('scans', 'qr-jump'); ?></label></p>
            <p><label><?php _e('Notification Email(s)', 'qr-jump'); ?> <input type="text" name="qr_jump_notify_emails" value="<?php echo esc_attr($notify_emails); ?>" class="regular-text"> <br><span class="description"><?php printf(__('Comma-separated. Defaults to admin email if blank.', 'qr-jump'), esc_html($global_emails)); ?></span></label></p>
        </div>
        <?php
    }

    public function save_meta_box_data($post_id) {
        if (!isset($_POST['qr_jump_meta_box_nonce'])) {
            return;
        }

        if (!wp_verify_nonce($_POST['qr_jump_meta_box_nonce'], 'qr_jump_meta_box')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        $fields = array(
            'qr_jump_destination' => 'text',
            'qr_jump_expiry_date' => 'text',
            'qr_jump_scan_limit' => 'int',
            'qr_jump_color' => 'text',
            'qr_jump_size' => 'int',
            // Per-QR notification fields
            'qr_jump_notify_enabled' => 'int',
            'qr_jump_notify_threshold' => 'int',
            'qr_jump_notify_emails' => 'text',
        );

        foreach ($fields as $field => $type) {
            if ($field === 'qr_jump_notify_enabled') {
                $value = isset($_POST[$field]) ? '1' : '0';
            } else {
                $value = isset($_POST[$field]) ? $_POST[$field] : '';
                if ($type === 'int') {
                    $value = $value !== '' ? intval($value) : '';
                } elseif ($type === 'text') {
                    if ($field === 'qr_jump_destination') {
                        // Special handling for destination field - preserve vCard formatting
                        $value = sanitize_textarea_field($value);
                    } else {
                        $value = sanitize_text_field($value);
                    }
                }
            }
            
            // For expiry_date, handle both setting and clearing
            if ($field === 'qr_jump_expiry_date') {
                if (!empty($value)) {
                    $timestamp = strtotime($value);
                    if ($timestamp) {
                        $value = date('Y-m-d H:i:s', $timestamp - (get_option('gmt_offset') * HOUR_IN_SECONDS));
                    }
                } else {
                    $value = ''; // Clear the expiry date if empty
                }
            }
            
            update_post_meta($post_id, '_' . $field, $value);
        }
    }

    public function set_custom_columns($columns) {
        // Set columns in desired order: Title, Destination, Total Clicks, Last Scan, Last Day, Last 7 Days, Last 30 Days, This Month, Categories
        $new_columns = array();
        $new_columns['title'] = __('Title');
        $new_columns['destination'] = __('Destination', 'qr-jump');
        $new_columns['total_clicks'] = __('Total Clicks', 'qr-jump');
        $new_columns['last_day'] = __('Last Day', 'qr-jump');
        $new_columns['last_7_days'] = __('Last 7 Days', 'qr-jump');
        $new_columns['last_30_days'] = __('Last 30 Days', 'qr-jump');
        $new_columns['this_month'] = __('This Month', 'qr-jump');
        $new_columns['last_scan'] = __('Last Scan', 'qr-jump');
        if (isset($columns['taxonomy-qr_category'])) {
            $new_columns['taxonomy-qr_category'] = $columns['taxonomy-qr_category'];
        }
        if (isset($columns['date'])) {
            $new_columns['date'] = $columns['date'];
        }
        return $new_columns;
    }

    public function custom_column_content($column, $post_id) {
        switch ($column) {
            case 'destination':
                $destination = get_post_meta($post_id, '_qr_jump_destination', true);
                if ($destination) {
                    if (strpos(trim($destination), 'BEGIN:VCARD') === 0) {
                        echo '<span class="vcard-indicator">📇 vCard Contact</span>';
                    } else {
                        echo '<a href="' . esc_url($destination) . '" target="_blank">' . esc_html($destination) . '</a>';
                    }
                }
                break;
            case 'total_clicks':
                global $wpdb;
                $table_name = $wpdb->prefix . 'qr_jump_analytics';
                $total_clicks = $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM {$table_name} WHERE qr_id = %d",
                    $post_id
                ));
                echo number_format_i18n($total_clicks);
                break;
            case 'last_day':
                $analytics = new QR_Jump_Analytics();
                echo number_format_i18n($analytics->get_scans_last_day($post_id));
                break;
            case 'last_7_days':
                $analytics = new QR_Jump_Analytics();
                echo number_format_i18n($analytics->get_scans_last_7_days($post_id));
                break;
            case 'last_30_days':
                $analytics = new QR_Jump_Analytics();
                echo number_format_i18n($analytics->get_scans_last_30_days($post_id));
                break;
            case 'this_month':
                $analytics = new QR_Jump_Analytics();
                echo number_format_i18n($analytics->get_scans_this_month($post_id));
                break;
            case 'last_scan':
                global $wpdb;
                $table_name = $wpdb->prefix . 'qr_jump_analytics';
                $last_scan = $wpdb->get_var($wpdb->prepare(
                    "SELECT scan_date FROM {$table_name} WHERE qr_id = %d ORDER BY scan_date DESC LIMIT 1",
                    $post_id
                ));
                if ($last_scan) {
                    $utc = new DateTime($last_scan, new DateTimeZone('UTC'));
                    $wp_tz = function_exists('wp_timezone') ? wp_timezone() : new DateTimeZone(wp_timezone_string());
                    $utc->setTimezone($wp_tz);
                    echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $utc->getTimestamp()));
                } else {
                    echo '<span class="no-scans">' . esc_html__('No scans yet', 'qr-jump') . '</span>';
                }
                break;
        }
    }

    public function add_analytics_meta_box() {
        add_meta_box(
            'qr_jump_analytics',
            __('Scan Analytics', 'qr-jump'),
            array($this, 'render_analytics_meta_box'),
            'qr_code',
            'normal',
            'high'
        );
    }

    public function render_analytics_meta_box($post) {
        // Show success message if redirected after reset
        if (isset($_GET['qr_jump_reset_scans_success']) && $_GET['qr_jump_reset_scans_success'] == 1) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('All scans for this QR code have been reset.', 'qr-jump') . '</p></div>';
        }
        $stats = $this->analytics->get_scan_stats($post->ID);
        $scans_last_day = $this->analytics->get_scans_last_day($post->ID);
        $scans_last_7 = $this->analytics->get_scans_last_7_days($post->ID);
        $scans_last_30 = $this->analytics->get_scans_last_30_days($post->ID);
        $scans_this_month = $this->analytics->get_scans_this_month($post->ID);
        ?>
        <div class="qr-jump-analytics">
            <div class="analytics-summary">
                <div class="stat-box">
                    <h4><?php _e('Total Scans', 'qr-jump'); ?></h4>
                    <p class="stat-number"><?php echo number_format_i18n($stats['total_scans']); ?></p>
                </div>
                <div class="stat-box">
                    <h4><?php _e('Unique Visitors', 'qr-jump'); ?></h4>
                    <p class="stat-number"><?php echo number_format_i18n($stats['unique_ips']); ?></p>
                </div>
                <div class="stat-box">
                    <h4><?php _e('Scans in the last day', 'qr-jump'); ?></h4>
                    <p class="stat-number"><?php echo esc_html($scans_last_day); ?></p>
                </div>
                <div class="stat-box">
                    <h4><?php _e('Scans in the last 7 days', 'qr-jump'); ?></h4>
                    <p class="stat-number"><?php echo esc_html($scans_last_7); ?></p>
                </div>
                <div class="stat-box">
                    <h4><?php _e('Scans in the last 30 days', 'qr-jump'); ?></h4>
                    <p class="stat-number"><?php echo esc_html($scans_last_30); ?></p>
                </div>
                <div class="stat-box">
                    <h4><?php _e('Scans this month', 'qr-jump'); ?></h4>
                    <p class="stat-number"><?php echo esc_html($scans_this_month); ?></p>
                </div>
            </div>

            <?php if ($stats['total_scans'] > 0) : ?>
                <div class="analytics-details">
                    <h3><?php _e('Recent Scans', 'qr-jump'); ?></h3>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php _e('Date/Time', 'qr-jump'); ?></th>
                                <th><?php _e('IP Address', 'qr-jump'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            global $wpdb;
                            $table_name = $wpdb->prefix . 'qr_jump_analytics';
                            $scans = $wpdb->get_results($wpdb->prepare(
                                "SELECT * FROM {$table_name} WHERE qr_id = %d ORDER BY scan_date DESC LIMIT 10",
                                $post->ID
                            ));

                            foreach ($scans as $scan) :
                                $utc = new DateTime($scan->scan_date, new DateTimeZone('UTC'));
                                $wp_tz = function_exists('wp_timezone') ? wp_timezone() : new DateTimeZone(wp_timezone_string());
                                $utc->setTimezone($wp_tz);
                                $formatted_date = date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $utc->getTimestamp());
                            ?>
                                <tr>
                                    <td><?php echo esc_html($formatted_date); ?></td>
                                    <td><?php echo esc_html($scan->ip_address); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>

                    <div class="analytics-actions">
                        <!-- <a href="<?php echo esc_url(admin_url('admin-ajax.php?action=qr_jump_export_csv&qr_id=' . $post->ID . '&nonce=' . wp_create_nonce('qr_jump_export'))); ?>" class="button">
                            <?php _e('Export All Scans to CSV', 'qr-jump'); ?>
                        </a> -->
                    </div>
                </div>
            <?php else : ?>
                <p><?php _e('No scans recorded yet.', 'qr-jump'); ?></p>
            <?php endif; ?>
        </div>
        <?php
    }

    public function render_qr_preview_meta_box($post) {
        // This function is now intentionally left blank (meta box not registered)
    }

    public function handle_qr_download() {
        if (!isset($_GET['qr_id']) || !isset($_GET['nonce']) || !isset($_GET['format'])) {
            wp_die(__('Invalid request.', 'qr-jump'));
        }

        if (!wp_verify_nonce($_GET['nonce'], 'qr_jump_download')) {
            wp_die(__('Security check failed.', 'qr-jump'));
        }

        $qr_id = intval($_GET['qr_id']);
        $format = sanitize_text_field($_GET['format']);
        $post = get_post($qr_id);
        if (!$post || $post->post_type !== 'qr_code') {
            wp_die(__('QR code not found.', 'qr-jump'));
        }
        $qr_color = get_post_meta($qr_id, '_qr_jump_color', true) ?: '#000000';
        $qr_size = get_post_meta($qr_id, '_qr_jump_size', true) ?: 300;
        $qr_url = home_url('qr/' . $post->post_name);
        if ($format === 'svg') {
            $qr_code = $this->qr_generator->generate_qr_code_svg($qr_url, array('color' => $qr_color, 'size' => $qr_size));
            $mime_type = 'image/svg+xml';
            $extension = 'svg';
        } else {
            // For PNG, get the raw binary data (not data URI)
            $qr_code = $this->qr_generator->generate_qr_code_file($qr_url, array('color' => $qr_color, 'size' => $qr_size));
            $mime_type = 'image/png';
            $extension = 'png';
        }
        if (!$qr_code) {
            wp_die(__('Error generating QR code.', 'qr-jump'));
        }
        $filename = 'qr-code-' . $qr_id . '.' . $extension;
        header('Content-Type: ' . $mime_type);
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($qr_code));
        header('Cache-Control: no-cache, must-revalidate');
        header('Pragma: no-cache');
        echo $qr_code;
        exit;
    }

    public function show_logo_on_qr_code_list() {
        $screen = get_current_screen();
        if ($screen && $screen->id === 'edit-qr_code') {
            echo '<div style="margin: 20px 0 10px 0;"><img src="' . plugin_dir_url(__FILE__) . 'css/qrjump-logo.svg" alt="QR Jump Logo" style="max-width:200px; display:block;"></div>';
        }
    }

    // Add this new method to show the logo above the meta box on QR Code edit/add screens
    public function show_logo_on_qr_code_edit() {
        $screen = get_current_screen();
        if ($screen && $screen->post_type === 'qr_code' && in_array($screen->base, ['post', 'post-new'])) {
            echo '<div style="margin: 20px 0 10px 0;"><img src="' . plugin_dir_url(__FILE__) . 'css/qrjump-logo.svg" alt="QR Jump Logo" style="max-width:200px; display:block;"></div>';
        }
    }
} 